open Formeltyp

type formel_symbol =
  | PlusSym | MinusSym | SternSym | DPunktSym | DachSym 
  | Hoch1Sym | Hoch2Sym | Hoch3Sym | KlammerAufSym | KlammerZuSym 
  | XSym 
  | ZahlLiteral of float
  | Ende
;;


let rec formel_lexer =
  lexer
    | '+' -> PlusSym
    | '-' -> MinusSym
    | '*' -> SternSym
    | ':' -> DPunktSym
    | '^' -> DachSym
    | '' -> Hoch1Sym
    | '' -> Hoch2Sym
    | '' -> Hoch3Sym
    | '(' -> KlammerAufSym
    | ')' -> KlammerZuSym
    | 'x' -> XSym
    | [ '0' - '9' ]+ ( '.' [ '0' - '9' ]* )? ->
	ZahlLiteral(float_of_string(Ulexing.latin1_lexeme lexbuf))
    | [ ' ' '\t' '\r' '\n' ]+ ->
	formel_lexer lexbuf
    | eof -> Ende
;;


type kette =
  | Einzel of formel
  | Kette of formel * formel_symbol * kette
;;


let rec formel_von_kette kette =
  match kette with
    | Einzel formel ->
	formel
    | Kette (formel1, sym, Einzel formel2) ->
	formel_fr_sym sym formel1 formel2
    | Kette (formel1, sym1, Kette(formel2, sym2, kette2)) ->
	(* Links-assoziative Interpretation einer Kette: *)
	formel_von_kette
	  (Kette(formel_fr_sym sym1 formel1 formel2,
		 sym2,
		 kette2))

and formel_fr_sym sym formel1 formel2 =
  match sym with
    | PlusSym   -> Add(formel1, formel2)
    | MinusSym  -> Sub(formel1, formel2)
    | SternSym  -> Mul(formel1, formel2)
    | DPunktSym -> Div(formel1, formel2)
    | DachSym   -> Pot(formel1, formel2)
    | _         -> failwith "formel_fr_sym"
;;


let rec formel_parser sym_stream =
  match sym_stream with parser
      [< formel = faktor_parser;
	 fortsetzung_opt = formel_fortsetzungs_parser;
      >] ->
	( match fortsetzung_opt with
	    | None -> formel
	    | Some (sym, kette) ->
		formel_von_kette (Kette (formel, sym, kette))
	)

and formel_fortsetzungs_parser sym_stream =
  match sym_stream with parser
    | [< '(PlusSym | MinusSym as sym);
	 formel = faktor_parser;
	 fortsetzung_opt = formel_fortsetzungs_parser
      >] ->
	( match fortsetzung_opt with
	    | None -> 
		Some(sym, Einzel formel)
	    | Some (sym', kette') ->
		Some(sym, Kette (formel, sym', kette'))
	)
    | [< >] ->
	None

and faktor_parser sym_stream =
  match sym_stream with parser
    | [< formel = potausdr_parser;
	 fortsetzung_opt = faktor_fortsetzungs_parser
      >] ->
	( match fortsetzung_opt with
	    | None -> formel
	    | Some (sym, kette) ->
		formel_von_kette (Kette (formel, sym, kette))
	)

and faktor_fortsetzungs_parser sym_stream =
  match sym_stream with parser
    | [< '(SternSym | DPunktSym as sym);
	 formel = potausdr_parser;
	 fortsetzung_opt = faktor_fortsetzungs_parser
      >] ->
	( match fortsetzung_opt with
	    | None -> 
		Some(sym, Einzel formel)
	    | Some (sym', kette') ->
		Some(sym, Kette (formel, sym', kette'))
	)
    | [< formel = potausdr_parser;
	 fortsetzung_opt = faktor_fortsetzungs_parser
      >] ->
	( match fortsetzung_opt with
	    | None -> 
		Some(SternSym, Einzel formel)
	    | Some (sym', kette') ->
		Some(SternSym, Kette (formel, sym', kette'))
	)
    | [< >] ->
	None

and potausdr_parser sym_stream =
  match sym_stream with parser
    | [< formel = atom_parser;
	 fortsetzung_opt = potausdr_fortsetzungs_parser
      >] ->
	( match fortsetzung_opt with
	    | None -> formel
	    | Some (sym, kette) ->
		formel_von_kette (Kette (formel, sym, kette))
	)

and potausdr_fortsetzungs_parser sym_stream =
  match sym_stream with parser
    | [< '(DachSym as sym);
	 formel = atom_parser;
	 fortsetzung_opt = potausdr_fortsetzungs_parser
      >] ->
	( match fortsetzung_opt with
	    | None -> 
		Some(sym, Einzel formel)
	    | Some (sym', kette') ->
		Some(sym, Kette (formel, sym', kette'))
	)

    | [< '(Hoch1Sym | Hoch2Sym | Hoch3Sym as sym);
	 fortsetzung_opt = potausdr_fortsetzungs_parser
      >] ->
	let formel =
	  match sym with
	    | Hoch1Sym -> Zahl 1.0
	    | Hoch2Sym -> Zahl 2.0
	    | Hoch3Sym -> Zahl 3.0
	    | _ -> assert false in
	( match fortsetzung_opt with
	    | None -> 
		Some(DachSym, Einzel formel)
	    | Some (sym', kette') ->
		Some(DachSym, Kette (formel, sym', kette'))
	)

    | [< >] ->
	None

and atom_parser sym_stream =
  match sym_stream with parser
    | [< 'ZahlLiteral z >] ->
	Zahl z
    | [< 'XSym >] ->
	X
    | [< 'KlammerAufSym;
	 formel = formel_parser;
	 'KlammerZuSym
      >] ->
	formel
;;


let formel_von_string s =
  try
    let lexbuf = Ulexing.from_latin1_string s in
    let sym_stream =
      Stream.from
	(fun _ -> 
	   match formel_lexer lexbuf with
	     | Ende -> None
	     | sym -> Some sym
	) in
    formel_parser sym_stream
  with
    | Ulexing.Error ->
	failwith "Unbekanntes Zeichen in Formel benutzt"
    | Stream.Error _ 
    | Stream.Failure ->
	failwith "Syntax-Fehler in Formel"
;;


let rec expr_von_formel loc formel =
  match formel with
    | Add(f1, f2) ->
	let e1 = expr_von_formel loc f1 in
	let e2 = expr_von_formel loc f2 in
	<:expr< Add $e1$ $e2$ >>
    | Sub(f1, f2) ->
	let e1 = expr_von_formel loc f1 in
	let e2 = expr_von_formel loc f2 in
	<:expr< Sub $e1$ $e2$ >>
    | Mul(f1, f2) ->
	let e1 = expr_von_formel loc f1 in
	let e2 = expr_von_formel loc f2 in
	<:expr< Mul $e1$ $e2$ >>
    | Div(f1, f2) ->
	let e1 = expr_von_formel loc f1 in
	let e2 = expr_von_formel loc f2 in
	<:expr< Div $e1$ $e2$ >>
    | Pot(f1, f2) ->
	let e1 = expr_von_formel loc f1 in
	let e2 = expr_von_formel loc f2 in
	<:expr< Pot $e1$ $e2$ >>
    | Zahl z ->
	let s = string_of_float z in
	<:expr< Zahl $flo:s$ >>
    | X ->
	<:expr< X >>
;;


let formel_expr s =
  let formel = formel_von_string s in
  let loc = (Lexing.dummy_pos, Lexing.dummy_pos) in
  expr_von_formel loc formel
;;


let formel_pat s =
  failwith "Formeln in Pattern werden nicht untersttzt" ;;

Quotation.add
  "formel"
  (Quotation.ExAst(formel_expr, formel_pat)) 
;;
